from fastapi import APIRouter, HTTPException
from typing import Dict
import logging
from models.combination import AnalyticsEvent
from services.analytics import AnalyticsService
import os
from motor.motor_asyncio import AsyncIOMotorClient
from dotenv import load_dotenv
from pathlib import Path

# Load environment variables
ROOT_DIR = Path(__file__).parent.parent
load_dotenv(ROOT_DIR / '.env')

router = APIRouter(prefix="/analytics", tags=["analytics"])
logger = logging.getLogger(__name__)

# Database connection
mongo_url = os.environ['MONGO_URL']
client = AsyncIOMotorClient(mongo_url)
db = client[os.environ['DB_NAME']]

analytics_service = AnalyticsService(db)

@router.get("/stats")
async def get_stats():
    """Get general analytics and statistics"""
    try:
        stats = await analytics_service.get_stats()
        return stats
    except Exception as e:
        logger.error(f"Error fetching stats: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to fetch statistics")

@router.post("/track")
async def track_event(event: AnalyticsEvent):
    """Track an analytics event"""
    try:
        event_id = await analytics_service.track_event(
            event=event.event,
            data=event.data,
            session_id=event.sessionId,
            user_id=event.userId
        )
        
        return {"success": True, "eventId": event_id}
        
    except Exception as e:
        logger.error(f"Error tracking event: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to track event")