from fastapi import APIRouter, HTTPException, Depends
from typing import List, Optional
import logging
from models.combination import CombinationRequest, CombinationResult, PopularCombination
from services.name_combiner import NameCombinerService
from services.analytics import AnalyticsService
from motor.motor_asyncio import AsyncIOMotorDatabase
import os
from motor.motor_asyncio import AsyncIOMotorClient
from dotenv import load_dotenv
from pathlib import Path

# Load environment variables
ROOT_DIR = Path(__file__).parent.parent
load_dotenv(ROOT_DIR / '.env')

router = APIRouter(prefix="/combinations", tags=["combinations"])
logger = logging.getLogger(__name__)

# Database connection
mongo_url = os.environ['MONGO_URL']
client = AsyncIOMotorClient(mongo_url)
db = client[os.environ['DB_NAME']]

# Services
name_combiner_service = NameCombinerService()
analytics_service = AnalyticsService(db)

@router.post("/generate", response_model=CombinationResult)
async def generate_combinations(request: CombinationRequest):
    """Generate name combinations using advanced algorithms"""
    try:
        # Validate input
        if not request.names or len(request.names) < 2:
            raise HTTPException(status_code=400, detail="At least 2 names are required")
        
        # Clean and validate names
        clean_names = [name.strip() for name in request.names if name.strip()]
        if len(clean_names) < 2:
            raise HTTPException(status_code=400, detail="At least 2 valid names are required")
        
        # Generate combinations
        result = name_combiner_service.generate_combinations(clean_names, request.category)
        
        # Track analytics
        await analytics_service.track_event(
            event="combination_generated",
            data={
                "names_count": len(clean_names),
                "category": request.category,
                "results_count": len(result["combinations"])
            },
            user_id=request.userId
        )
        
        # Save successful combinations to database for popularity tracking
        if result["combinations"]:
            for combination in result["combinations"][:5]:  # Save top 5 results
                await analytics_service.save_combination(
                    original_names=clean_names,
                    result=combination,
                    category=request.category,
                    algorithm=result["metadata"].get("algorithm", "unknown"),
                    user_id=request.userId
                )
        
        logger.info(f"Generated {len(result['combinations'])} combinations for {len(clean_names)} names")
        return CombinationResult(**result)
        
    except Exception as e:
        logger.error(f"Error generating combinations: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to generate combinations")

@router.get("/popular", response_model=List[PopularCombination])
async def get_popular_combinations(category: Optional[str] = None, limit: int = 10):
    """Get popular combinations based on usage"""
    try:
        if limit > 50:
            limit = 50
            
        popular = await analytics_service.get_popular_combinations(category, limit)
        
        return [PopularCombination(**combo) for combo in popular]
        
    except Exception as e:
        logger.error(f"Error fetching popular combinations: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to fetch popular combinations")

@router.get("/categories")
async def get_categories():
    """Get available combination categories with descriptions"""
    categories = [
        {
            "name": "general",
            "displayName": "General",
            "description": "Standard name combinations for any purpose",
            "examples": ["John + Mary = Jary, Mohn"]
        },
        {
            "name": "couple",
            "displayName": "Couple Names",
            "description": "Romantic ship names perfect for couples",
            "examples": ["Brad + Angelina = Brangelina"]
        },
        {
            "name": "business",
            "displayName": "Business Names", 
            "description": "Professional brand names for companies",
            "examples": ["Tech + Innovation = Techovation"]
        },
        {
            "name": "baby",
            "displayName": "Baby Names",
            "description": "Family-friendly names for babies",
            "examples": ["David + Emma = Demma, Davem"]
        },
        {
            "name": "gaming",
            "displayName": "Gaming Names",
            "description": "Unique usernames and character names",
            "examples": ["Shadow + Warrior = Sharrior"]
        },
        {
            "name": "team",
            "displayName": "Team Names",
            "description": "Group names for teams and organizations",
            "examples": ["Thunder + Eagles = Thungles"]
        }
    ]
    
    return {"categories": categories}