from motor.motor_asyncio import AsyncIOMotorCollection
from typing import Dict, List, Optional
from datetime import datetime, timedelta
import uuid

class AnalyticsService:
    def __init__(self, db):
        self.analytics_collection: AsyncIOMotorCollection = db.analytics
        self.combinations_collection: AsyncIOMotorCollection = db.combinations
    
    async def track_event(self, event: str, data: Dict, session_id: Optional[str] = None, user_id: Optional[str] = None):
        """Track an analytics event"""
        event_doc = {
            "id": str(uuid.uuid4()),
            "event": event,
            "data": data,
            "sessionId": session_id,
            "userId": user_id,
            "timestamp": datetime.utcnow()
        }
        
        await self.analytics_collection.insert_one(event_doc)
        return event_doc["id"]
    
    async def get_popular_combinations(self, category: Optional[str] = None, limit: int = 10) -> List[Dict]:
        """Get popular combinations based on usage"""
        match_filter = {"isPublic": True}
        if category:
            match_filter["category"] = category
            
        pipeline = [
            {"$match": match_filter},
            {"$group": {
                "_id": {
                    "result": "$result",
                    "originalNames": "$originalNames",
                    "category": "$category"
                },
                "count": {"$sum": 1}
            }},
            {"$sort": {"count": -1}},
            {"$limit": limit},
            {"$project": {
                "name": "$_id.result",
                "originalNames": "$_id.originalNames",
                "category": "$_id.category",
                "count": 1,
                "_id": 0
            }}
        ]
        
        cursor = self.combinations_collection.aggregate(pipeline)
        return await cursor.to_list(length=limit)
    
    async def get_stats(self) -> Dict:
        """Get general statistics"""
        total_combinations = await self.combinations_collection.count_documents({})
        
        # Get category distribution
        category_pipeline = [
            {"$group": {"_id": "$category", "count": {"$sum": 1}}},
            {"$sort": {"count": -1}}
        ]
        
        category_cursor = self.combinations_collection.aggregate(category_pipeline)
        categories = await category_cursor.to_list(length=10)
        
        # Recent activity (last 24 hours)
        recent_cutoff = datetime.utcnow() - timedelta(days=1)
        recent_count = await self.combinations_collection.count_documents({
            "createdAt": {"$gte": recent_cutoff}
        })
        
        return {
            "totalCombinations": total_combinations,
            "recentCombinations": recent_count,
            "popularCategories": {cat["_id"]: cat["count"] for cat in categories}
        }
    
    async def save_combination(self, original_names: List[str], result: str, category: str, 
                             algorithm: str, user_id: Optional[str] = None) -> str:
        """Save a generated combination to database"""
        combination_doc = {
            "id": str(uuid.uuid4()),
            "originalNames": original_names,
            "result": result,
            "category": category,
            "algorithm": algorithm,
            "createdAt": datetime.utcnow(),
            "userId": user_id,
            "isPublic": True,
            "popularity": 0
        }
        
        result = await self.combinations_collection.insert_one(combination_doc)
        return combination_doc["id"]