import re
import random
from typing import List, Dict, Tuple
import itertools

class NameCombinerService:
    def __init__(self):
        self.vowels = set('aeiouAEIOU')
        self.consonants = set('bcdfghjklmnpqrstvwxyzBCDFGHJKLMNPQRSTVWXYZ')
        
    def generate_combinations(self, names: List[str], category: str = "general") -> Dict:
        """Generate name combinations using multiple algorithms"""
        if len(names) < 2:
            return {"combinations": [], "metadata": {}}
            
        clean_names = [name.strip().title() for name in names if name.strip()]
        
        if len(clean_names) < 2:
            return {"combinations": [], "metadata": {}}
            
        combinations = set()
        algorithms_used = []
        
        # Generate combinations for all pairs
        for i, name1 in enumerate(clean_names):
            for j, name2 in enumerate(clean_names):
                if i != j:
                    pair_combos = self._generate_pair_combinations(name1, name2, category)
                    combinations.update(pair_combos)
        
        # If more than 2 names, add multi-name combinations
        if len(clean_names) > 2:
            multi_combos = self._generate_multi_name_combinations(clean_names)
            combinations.update(multi_combos)
            
        # Filter and score results
        filtered_combinations = self._filter_and_score(list(combinations), category)
        
        # Sort by score and take top results
        filtered_combinations.sort(key=lambda x: x[1], reverse=True)
        final_results = [combo[0] for combo in filtered_combinations[:20]]
        
        metadata = {
            "algorithm": "advanced_multi_method",
            "category": category,
            "total_generated": len(combinations),
            "filtered_count": len(final_results),
            "average_score": sum(x[1] for x in filtered_combinations[:10]) / min(10, len(filtered_combinations)) if filtered_combinations else 0
        }
        
        return {
            "combinations": final_results,
            "metadata": metadata
        }
    
    def _generate_pair_combinations(self, name1: str, name2: str, category: str) -> List[str]:
        """Generate combinations for a pair of names"""
        combinations = []
        
        # Method 1: Syllable blending
        combinations.extend(self._syllable_blend(name1, name2))
        
        # Method 2: Prefix + Suffix combinations
        combinations.extend(self._prefix_suffix_blend(name1, name2))
        
        # Method 3: Vowel harmony
        combinations.extend(self._vowel_harmony_blend(name1, name2))
        
        # Method 4: Phonetic fusion
        combinations.extend(self._phonetic_fusion(name1, name2))
        
        # Method 5: Alternating pattern
        combinations.extend(self._alternating_pattern(name1, name2))
        
        # Category-specific enhancements
        if category == "couple":
            combinations.extend(self._romantic_blend(name1, name2))
        elif category == "business":
            combinations.extend(self._professional_blend(name1, name2))
        
        return combinations
    
    def _syllable_blend(self, name1: str, name2: str) -> List[str]:
        """Blend names at syllable boundaries"""
        combinations = []
        
        # Simple syllable detection based on vowel groups
        syllables1 = self._detect_syllables(name1)
        syllables2 = self._detect_syllables(name2)
        
        # Combine first part of name1 with last part of name2
        for i in range(1, len(syllables1)):
            for j in range(len(syllables2)):
                combo1 = ''.join(syllables1[:i]) + ''.join(syllables2[j:])
                combo2 = ''.join(syllables2[:j+1]) + ''.join(syllables1[i:])
                if 3 <= len(combo1) <= 12:
                    combinations.append(combo1)
                if 3 <= len(combo2) <= 12:
                    combinations.append(combo2)
        
        return combinations
    
    def _detect_syllables(self, name: str) -> List[str]:
        """Simple syllable detection"""
        # Split on vowel groups
        vowel_pattern = r'[aeiouAEIOU]+'
        parts = re.split(f'({vowel_pattern})', name)
        
        syllables = []
        current = ""
        
        for i, part in enumerate(parts):
            current += part
            if re.match(vowel_pattern, part) and i < len(parts) - 1:
                syllables.append(current)
                current = ""
        
        if current:
            syllables.append(current)
            
        return syllables if syllables else [name]
    
    def _prefix_suffix_blend(self, name1: str, name2: str) -> List[str]:
        """Combine prefixes and suffixes"""
        combinations = []
        
        for i in range(2, min(len(name1), 6)):
            for j in range(2, min(len(name2), 6)):
                # First i letters of name1 + last j letters of name2
                combo1 = name1[:i] + name2[-j:]
                # First j letters of name2 + last i letters of name1
                combo2 = name2[:j] + name1[-i:]
                
                if 3 <= len(combo1) <= 12:
                    combinations.append(combo1)
                if 3 <= len(combo2) <= 12:
                    combinations.append(combo2)
        
        return combinations
    
    def _vowel_harmony_blend(self, name1: str, name2: str) -> List[str]:
        """Replace vowels for better harmony"""
        combinations = []
        
        # Extract vowel patterns
        vowels1 = [c for c in name1.lower() if c in 'aeiou']
        vowels2 = [c for c in name2.lower() if c in 'aeiou']
        
        if vowels1 and vowels2:
            # Replace vowels in name1 with vowels from name2
            result1 = name1.lower()
            result2 = name2.lower()
            
            vowel_idx = 0
            for i, char in enumerate(result1):
                if char in 'aeiou' and vowel_idx < len(vowels2):
                    result1 = result1[:i] + vowels2[vowel_idx % len(vowels2)] + result1[i+1:]
                    vowel_idx += 1
            
            vowel_idx = 0
            for i, char in enumerate(result2):
                if char in 'aeiou' and vowel_idx < len(vowels1):
                    result2 = result2[:i] + vowels1[vowel_idx % len(vowels1)] + result2[i+1:]
                    vowel_idx += 1
            
            combinations.extend([result1.title(), result2.title()])
        
        return combinations
    
    def _phonetic_fusion(self, name1: str, name2: str) -> List[str]:
        """Fusion based on phonetic similarity"""
        combinations = []
        
        # Overlap blending - find common endings/beginnings
        for i in range(1, min(4, len(name1), len(name2))):
            if name1[-i:].lower() == name2[:i].lower():
                # Names overlap
                combo = name1 + name2[i:]
                if 3 <= len(combo) <= 12:
                    combinations.append(combo)
            
            if name2[-i:].lower() == name1[:i].lower():
                combo = name2 + name1[i:]
                if 3 <= len(combo) <= 12:
                    combinations.append(combo)
        
        return combinations
    
    def _alternating_pattern(self, name1: str, name2: str) -> List[str]:
        """Alternate letters from both names"""
        combinations = []
        
        # Only for shorter names to avoid too long results
        if len(name1) <= 6 and len(name2) <= 6:
            combo1 = ""
            combo2 = ""
            
            max_len = max(len(name1), len(name2))
            for i in range(max_len):
                if i < len(name1):
                    combo1 += name1[i]
                if i < len(name2):
                    combo1 += name2[i]
                    
                if i < len(name2):
                    combo2 += name2[i]
                if i < len(name1):
                    combo2 += name1[i]
            
            if 3 <= len(combo1) <= 12:
                combinations.append(combo1)
            if 3 <= len(combo2) <= 12 and combo2 != combo1:
                combinations.append(combo2)
        
        return combinations
    
    def _romantic_blend(self, name1: str, name2: str) -> List[str]:
        """Special blending for couple names"""
        combinations = []
        
        # Classic ship name style (first part + second part)
        if len(name1) >= 3 and len(name2) >= 3:
            # Take 2-4 letters from first name + 2-4 from second
            for i in range(2, min(5, len(name1) + 1)):
                for j in range(2, min(5, len(name2) + 1)):
                    combo = name1[:i] + name2[-j:].lower()
                    if 4 <= len(combo) <= 10:
                        combinations.append(combo.title())
        
        return combinations
    
    def _professional_blend(self, name1: str, name2: str) -> List[str]:
        """Professional-sounding business name combinations"""
        combinations = []
        
        # Add common business suffixes/prefixes conceptually
        business_patterns = [
            (name1[:3] + name2[:3], "Pro"),
            (name1[:4] + name2[-3:], "Tech"),
            (name2[:3] + name1[-4:], "Corp")
        ]
        
        for base, suffix in business_patterns:
            if 5 <= len(base) <= 8:
                combinations.extend([base, base + suffix])
        
        return combinations
    
    def _generate_multi_name_combinations(self, names: List[str]) -> List[str]:
        """Generate combinations from 3+ names"""
        combinations = []
        
        # Initials combination
        initials = ''.join(name[0].upper() for name in names)
        combinations.append(initials)
        
        # First two letters of each name
        if all(len(name) >= 2 for name in names):
            two_letter_combo = ''.join(name[:2] for name in names)
            if len(two_letter_combo) <= 12:
                combinations.append(two_letter_combo.title())
        
        # Base name + additions
        if len(names) >= 3:
            base = names[0]
            additions = ''.join(name[:2] for name in names[1:3])
            combo = base + additions.lower()
            if len(combo) <= 12:
                combinations.append(combo)
        
        return combinations
    
    def _filter_and_score(self, combinations: List[str], category: str) -> List[Tuple[str, float]]:
        """Filter bad combinations and score the good ones"""
        scored_combinations = []
        
        for combo in set(combinations):  # Remove duplicates
            if not combo or len(combo) < 3:
                continue
                
            score = self._calculate_score(combo, category)
            if score > 0.3:  # Minimum quality threshold
                scored_combinations.append((combo, score))
        
        return scored_combinations
    
    def _calculate_score(self, combination: str, category: str) -> float:
        """Calculate quality score for a combination"""
        score = 1.0
        
        # Length penalty/bonus
        length = len(combination)
        if 4 <= length <= 8:
            score += 0.2
        elif length <= 3 or length > 12:
            score -= 0.3
        
        # Vowel/consonant balance
        vowel_count = sum(1 for c in combination.lower() if c in 'aeiou')
        consonant_count = len(combination) - vowel_count
        
        if vowel_count == 0 or consonant_count == 0:
            score -= 0.5
        else:
            ratio = min(vowel_count, consonant_count) / max(vowel_count, consonant_count)
            score += ratio * 0.2
        
        # Avoid awkward letter combinations
        awkward_patterns = ['xxx', 'yyy', 'zzz', 'qx', 'xq', 'zx']
        for pattern in awkward_patterns:
            if pattern in combination.lower():
                score -= 0.3
        
        # Category-specific scoring
        if category == "business":
            # Professional sounding bonus
            if combination[0].isupper() and combination[1:].islower():
                score += 0.1
        elif category == "couple":
            # Romantic flow bonus (ends with vowel)
            if combination[-1].lower() in 'aeiou':
                score += 0.1
        
        # Pronounceability bonus (alternating vowels/consonants somewhat)
        pronounce_score = 0
        for i in range(len(combination) - 1):
            current_is_vowel = combination[i].lower() in 'aeiou'
            next_is_vowel = combination[i + 1].lower() in 'aeiou'
            if current_is_vowel != next_is_vowel:
                pronounce_score += 1
        
        if len(combination) > 1:
            score += (pronounce_score / (len(combination) - 1)) * 0.2
        
        return max(0.0, min(1.0, score))